/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.security.action;

import java.security.*;

/**
 * A convenience class for retrieving the string value of a system property
 * as a privileged action.
 *
 * @see java.security.PrivilegedAction
 * @see java.security.AccessController
 */
public class GetPropertyAction implements PrivilegedAction {
    private String prop;
    private String defaultVal;

    /**
     * Constructor that takes the name of the system property whose
     * string value needs to be determined.
     *
     * @param prop the name of the system property
     */
    public GetPropertyAction(String prop) {
        this(prop, null);
    }

    /**
     * Constructor that takes the name of the system property and the default
     * value to be used when property is not set.
     *
     * @param prop the name of the system property
     * @param defaultVal the default value
     */
    public GetPropertyAction(String prop, String defaultVal) {
        this.prop = prop;
        this.defaultVal = defaultVal;
    }

    /**
     * Determines the string value of the system property whose
     * name was specified in the constructor.
     *
     * @return the string value of the system property,
     *         or the default value if there is no property with that key.
     */
    public Object run() {
        String val = System.getProperty(prop);
        return (val != null) ? val : defaultVal;
    }
}